﻿#requires -version 3
Set-StrictMode -Version Latest

# Set the default cache time to 60 minutes. Less than 10 minutes is against their wishes,
# so I prevent that in Set-CacheMinutes (almost impossible to crack this!).
$script:CacheMinutes = 20

# Cache hash, indexed on city IDs (presumably they are unique).
$script:Cache = @{}

# Hard-coded URI bases
$script:CitySearchUri = 'http://api.openweathermap.org/data/2.5/weather?q=' #"http://api.openweathermap.org/data/2.1/find/name?q="
#$script:CityInfoUriBase = 'http://api.openweathermap.org/data/2.1/weather/city/'

function Get-OWMCacheMinutes {
    $script:CacheMinutes
}

function Set-OWMCacheMinutes {
    
    param([Parameter(Mandatory)][int] $Minutes)
    
    if ($Minutes -lt 10) {
        throw "Minutes value needs to be 10 or higher."
    }
    
    $script:CacheMinutes = $Minutes
    
}

function Get-OWMCache {
    
    if (-not $script:Cache.Count) { "Cache is empty."; return }
    $script:Cache.GetEnumerator() | Sort Name

}

function Write-MyError {
    
    param(
        [Parameter(Mandatory)][string] $Output,
        [switch] $Object
    )

    if ($Object) {
        New-Object PSObject -Property @{ Error = $Output }
    }
    else {
        $Output
    }

}

<#
.SYNOPSIS
Svendsen Tech's OpenWeatherMap module.
This Get-OWMWeather gets weather data for a city string using the OpenWeatherMap API v2.5.

Date: 2014-06-22
Copyright (c) 2013, Svendsen Tech. All rights reserved.

.DESCRIPTION
The Get-OWMWeather cmdlet gets weather data for a city string using the OpenWeatherMap API v2.5.

.EXAMPLE
PS D:\PS> Get-OWMWeather -City 'london,uk' -Object

coord   : @{lon=-0.13; lat=51.51}
sys     : @{message=0.2092; country=GB; sunrise=1403408603; sunset=1403468509}
weather : {@{id=802; main=Clouds; description=scattered clouds; icon=03n}}
base    : cmc stations
main    : @{temp=287.76; pressure=1010; temp_min=285.93; temp_max=289.82; humidity=75}
wind    : @{speed=1.81; deg=0.00396729}
clouds  : @{all=36}
dt      : 1403399701
id      : 2643743
name    : London
cod     : 200

.EXAMPLE
PS D:\PowerShell> Get-OWMWeather -City 'london,uk'
Place: London. Temperature 14.61C / 58.30F. Humidity: 75 %. Pressure: 1010 hpa. Date: 2014-06-22 01:15:01. Country: GB.
 Wind speed: 1.81 m/s. Weather: scattered clouds. Sunrise: 2014-06-22 03:43:23. Sunset: 2014-06-22 20:21:49

PS D:\PowerShell> 'london,uk' | Get-OWMWeather
Place: London. Temperature 14.61C / 58.30F. Humidity: 75 %. Pressure: 1010 hpa. Date: 2014-06-22 01:15:01. Country: GB.
 Wind speed: 1.81 m/s. Weather: scattered clouds. Sunrise: 2014-06-22 03:43:23. Sunset: 2014-06-22 20:21:49 (cached; wi
ll refresh in 20 minutes)

.PARAMETER City
Required. City search string.
.PARAMETER Object
Optional. Returns the full JSON object rather than a formatted weather string.
.PARAMETER NoAPIKey
Optional. Makes the module not complain about a missing API key. The API is currently
"somewhat available" without an API key.
#>

function Get-OWMWeather {
    
    [CmdletBinding()]
    param(
        [Parameter(Mandatory,ValueFromPipeline)][string[]] $City,
        [switch] $Object,
        [switch] $NoAPIKey
    )
    
    if (-not (Get-Variable -Scope Global -Name OWMAPIKey -EA SilentlyContinue) -and -not $NoAPIKey) {
        throw "You need to set the global variable `$Global:OWMAPIKey or to try with -NoAPIKey."
    }
    
    #Add-Type -AssemblyName System.Web # added to manifest file
    
    foreach ($IterCity in $City) {
        
        $Uri = $script:CitySearchUri + [Web.HttpUtility]::UrlEncode($IterCity) + $(if ($NoAPIKey) { '' } else { "&APPID=$global:OWMAPIKey" })
    
        try {
            $t = Invoke-RestMethod -Uri $Uri
        }
        catch {
            Write-MyError -Output "Unable to execute Invoke-RestMethod to retrieve ID(s) for ${City}: $_" -Object:$Object
            return
        }
        
        if (($t | Get-Member -Name Message -EA 0) -and $t.Message -ilike '*Not found*') {
            Write-MyError -output "Did not find $IterCity." -Object:$Object
            return
        }
        
        if ($script:Cache.ContainsKey($IterCity)) {
            if (((Get-Date) - $script:Cache.$IterCity.Date).TotalMinutes -lt $script:CacheMinutes ) {
                $RefreshMins = [int]($script:CacheMinutes - ((Get-Date) - $script:Cache.$IterCity.Date).TotalMinutes)
                if ($Object) {
                    $script:Cache.$IterCity.WeatherObject
                }
                else {
                    $script:Cache.$IterCity.WeatherString + " (cached; will refresh in $RefreshMins minutes)"
                }
                continue
            }
        }
        
        # Get both C and F for temperatures
        if ($Kelvin = ($t.main.temp)) {
            $CelsiusTemp = $Kelvin - 273.15
            $FahrenheitTemp = $CelsiusTemp * 9/5 + 32
        }
        else {
            $CelsiusTemp = 'Unknown '
            $FahrenheitTemp = 'Unknown '
        }
        
        $IterCityData = @(
            $t.name,
            $CelsiusTemp,
            $FahrenheitTemp,
            $t.main.humidity,
            $t.main.pressure,
            (Get-Date -Date '1970-01-01T00:00:00').AddSeconds($t.dt).ToString('yyyy-MM-dd HH:mm:ss'),
            $t.sys.country,
            $t.wind.speed,
            $t.weather.description,
            (Get-Date -Date '1970-01-01T00:00:00').AddSeconds($t.sys.sunrise).ToString('yyyy-MM-dd HH:mm:ss'),
            (Get-Date -Date '1970-01-01T00:00:00').AddSeconds($t.sys.sunset).ToString('yyyy-MM-dd HH:mm:ss')
            #$(if ($t = ($IterCityJson | Select -EA SilentlyContinue -Exp url)) { $t } else { 'Unknown' })
        )
        
        if ($Object) {
            $t
        }
        else {
            "Place: {0}. Temperature {1:N2}C / {2:N2}F. Humidity: {3} %. Pressure: {4} hpa. Date: {5}. Country: {6}. Wind speed: {7} m/s. Weather: {8}. Sunrise: {9}. Sunset: {10}" -f $IterCityData
        }
        # Store data in cache
        $Script:Cache.$IterCity = New-Object PSObject -Property @{
            'WeatherObject' = $t
            'WeatherString' = "Place: {0}. Temperature {1:N2}C / {2:N2}F. Humidity: {3} %. Pressure: {4} hpa. Date: {5}. Country: {6}. Wind speed: {7} m/s. Weather: {8}. Sunrise: {9}. Sunset: {10}" -f $IterCityData
            'Date' = Get-Date
        }
            
        #Start-Sleep -Seconds 3

    } # end of foreach $IterCity

}

#$Global:OWMAPIKey = ''

Export-ModuleMember -Function Get-OWMCacheMinutes, Set-OWMCacheMinutes, Get-OWMWeather, Get-OWMCache

